<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Subscriptions;
use App\Models\AdminSettings;
use App\Models\Withdrawals;
use App\Models\Updates;
use App\Models\Like;
use App\Models\Notifications;
use App\Models\Reports;
use App\Models\PaymentGateways;
use App\Models\Transactions;
use App\Models\VerificationRequests;
use App\Models\Deposits;
use App\Helper;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Yabacon\Paystack;
use Image;
use DB;

class UserController extends Controller
{
  use Traits\UserDelete;
  use Traits\Functions;

  public function __construct(Request $request, AdminSettings $settings) {
    $this->request = $request;
    $this->settings = $settings::first();
  }

  /**
	 * Display dashboard user
	 *
	 * @return Response
	 */
  public function dashboard()
  {
    $earningNetUser = auth()->user()->myPaymentsReceived()->sum('earning_net_user');

    $subscriptionsActive = auth()->user()
      ->mySubscriptions()
        ->where('stripe_id', '=', '')
          ->whereDate('ends_at', '>=', Carbon::today())
          ->orWhere('stripe_status', 'active')
            ->where('stripe_id', '<>', '')
              ->whereStripePlan(auth()->user()->plan)
              ->orWhere('stripe_id', '=', '')
            ->where('stripe_plan', auth()->user()->plan)
        ->where('free', '=', 'yes')
      ->count();

    $month = date('m');
    $year = date('Y');
    $daysMonth = Helper::daysInMonth($month, $year);
    $dateFormat = "$year-$month-";

    $monthFormat  = trans("months.$month");
    $currencySymbol = $this->settings->currency_symbol;

    for ($i=1; $i <= $daysMonth; ++$i) {

      $date = date('Y-m-d', strtotime($dateFormat.$i));
      $_subscriptions = auth()->user()->myPaymentsReceived()->whereDate('created_at', '=', $date)->sum('earning_net_user');

      $monthsData[] =  "'$monthFormat $i'";


      $_earningNetUser = $_subscriptions;

      $earningNetUserSum[] = $_earningNetUser;

    }

		// Today
		$stat_revenue_today = Transactions::where('created_at', '>=', date('Y-m-d H:i:s', strtotime('today')))
		->whereApproved('1')
    ->whereSubscribed(auth()->user()->id)
		 ->sum('earning_net_user');

     // Yesterday
 		$stat_revenue_yesterday = Transactions::where('created_at', '>=', Carbon::yesterday())
 		->whereApproved('1')
     ->whereSubscribed(auth()->user()->id)
 		 ->sum('earning_net_user');

		 // Week
	 	$stat_revenue_week = Transactions::whereBetween('created_at', [
	        Carbon::parse()->startOfWeek(),
	        Carbon::parse()->endOfWeek(),
	    ])->whereApproved('1')
      ->whereSubscribed(auth()->user()->id)
	 	 ->sum('earning_net_user');

     // Last Week
	 	$stat_revenue_last_week = Transactions::whereBetween('created_at', [
	        Carbon::now()->startOfWeek()->subWeek(),
	        Carbon::now()->subWeek()->endOfWeek(),
	    ])->whereApproved('1')
      ->whereSubscribed(auth()->user()->id)
	 	 ->sum('earning_net_user');

		 // Month
	 	$stat_revenue_month = Transactions::whereBetween('created_at', [
	        Carbon::parse()->startOfMonth(),
	        Carbon::parse()->endOfMonth(),
	    ])->whereApproved('1')
      ->whereSubscribed(auth()->user()->id)
			->sum('earning_net_user');

      // Last Month
 	 	$stat_revenue_last_month = Transactions::whereBetween('created_at', [
 	        Carbon::now()->startOfMonth()->subMonth(),
 	        Carbon::now()->subMonth()->endOfMonth(),
 	    ])->whereApproved('1')
       ->whereSubscribed(auth()->user()->id)
 			->sum('earning_net_user');

    $label = implode(',', $monthsData);
    $data = implode(',', $earningNetUserSum);

    return view('users.dashboard', [
          'earningNetUser' => $earningNetUser,
          'subscriptionsActive' => $subscriptionsActive,
          'label' => $label,
          'data' => $data,
          'month' => $monthFormat,
          'stat_revenue_today' => $stat_revenue_today,
          'stat_revenue_yesterday' => $stat_revenue_yesterday,
    			'stat_revenue_week' => $stat_revenue_week,
          'stat_revenue_last_week' => $stat_revenue_last_week,
    			'stat_revenue_month' => $stat_revenue_month,
          'stat_revenue_last_month' => $stat_revenue_last_month
        ]);
  }

  public function profile($slug, $media = null)
  {

    $media = request('media');
    $mediaTitle = null;
    $sortPostByTypeMedia = null;

    if (isset($media)) {
      $mediaTitle = trans('general.'.$media.'').' - ';
      $sortPostByTypeMedia = '&media='.$media;
      $media = '/'.$media;
    }

    // All Payments
    $allPayment = PaymentGateways::where('enabled', '1')->whereSubscription('yes')->get();

    // Stripe Key
      $_stripe = PaymentGateways::whereName('Stripe')->where('enabled', '1')->select('key')->first();

      $user    = User::where('username','=', $slug)->where('status','active')->firstOrFail();

      // Hidden Profile Admin
      if (auth()->check() && $this->settings->hide_admin_profile == 'on' && $user->id == 1 && auth()->user()->id != 1) {
        abort(404);
      } elseif (auth()->guest() && $this->settings->hide_admin_profile == 'on' && $user->id == 1) {
        abort(404);
      }

      if (isset($media)) {
        $query = $user->updates();
      } else {
        $query = $user->updates()->whereFixedPost('0');
      }

      //=== Photos
  		$query->when(request('media') == 'photos', function($q) {
  			$q->where('image', '<>', '');
  		});

      //=== Videos
  		$query->when(request('media') == 'videos', function($q) use($user) {
  			$q->where('video', '<>', '')->orWhere('video_embed', '<>', '')->whereUserId($user->id);
  		});

      //=== Audio
  		$query->when(request('media') == 'audio', function($q) {
  			$q->where('music', '<>', '');
  		});

      //=== Files
  		$query->when(request('media') == 'files', function($q) {
  			$q->where('file', '<>', '');
  		});

      $updates = $query->orderBy('id','desc')->paginate($this->settings->number_posts_show);

      // Check if subscription exists
      if (auth()->check()) {
        $checkSubscription = auth()->user()->checkSubscription($user);

        if ($checkSubscription) {
          // Get Payment gateway the subscription
          $paymentGatewaySubscription = Transactions::whereSubscriptionsId($checkSubscription->id)->first();
        }

        // Check Payment Incomplete
        $paymentIncomplete = auth()->user()
          ->userSubscriptions()
            ->where('stripe_plan', $user->plan)
            ->whereStripeStatus('incomplete')
            ->first();
      }

      if ($user->status == 'suspended') {
        abort(404);
      }

      //<<<-- * Redirect the user real name * -->>>
      $uri = request()->path();
      $uriCanonical = $user->username.$media;

      if ($uri != $uriCanonical) {
        return redirect($uriCanonical);
      }

      // Find post pinned
      $findPostPinned = $user->updates()->whereFixedPost('1')->paginate($this->settings->number_posts_show);

      // Count all likes
      $likeCount = $user->likesCount();

      // Subscription sActive
      $subscriptionsActive = $user->mySubscriptions()
          ->where('stripe_id', '=', '')
            ->whereDate('ends_at', '>=', Carbon::today())
            ->orWhere('stripe_status', 'active')
              ->where('stripe_id', '<>', '')
                ->whereStripePlan($user->plan)
                ->orWhere('stripe_id', '=', '')
              ->where('stripe_plan', $user->plan)
          ->where('free', '=', 'yes')
        ->count();

      return view('users.profile',[
          'user' => $user,
            'updates' => $updates,
            'findPostPinned' => $findPostPinned,
            '_stripe' => $_stripe,
            'checkSubscription' => $checkSubscription ?? null,
            'media' => $media,
            'mediaTitle' => $mediaTitle,
            'sortPostByTypeMedia' => $sortPostByTypeMedia,
            'allPayment' => $allPayment,
            'paymentIncomplete' => $paymentIncomplete ?? null,
            'likeCount' => $likeCount,
            'paymentGatewaySubscription' => $paymentGatewaySubscription->payment_gateway ?? null,
            'subscriptionsActive' => $subscriptionsActive
        ]);

  }//<--- End Method

  public function postDetail($slug, $id)
  {

      $user    = User::where( 'username','=', $slug )->where('status','active')->firstOrFail();
      $updates = $user->updates()->whereId($id)->orderBy('id','desc')->paginate(1);

      $users = $this->userExplore();

      if ($user->status == 'suspended' || $updates->count() == 0) {
        abort(404);
      }

      //<<<-- * Redirect the user real name * -->>>
      $uri = request()->path();
      $uriCanonical = $user->username.'/post/'.$updates[0]->id;

      if( $uri != $uriCanonical ) {
        return redirect($uriCanonical);
      }

      return view('users.post-detail',
          ['user' => $user,
          'updates' => $updates,
          'inPostDetail' => true,
          'users' => $users
        ]);

  }//<--- End Method


    public function settings()
    {
        return view('users.settings');
    }

    public function updateSettings()
    {
      $input = $this->request->all();
      $id = auth()->user()->id;

     $validator = Validator::make($input, [
    'profession'  => 'required|min:6|max:100|string',
    'countries_id' => 'required',
    ]);

     if ($validator->fails()) {
         return redirect()->back()
                   ->withErrors($validator)
                   ->withInput();
     }

     $user               = User::find($id);
     $user->profession   = trim(strip_tags($input['profession']));
     $user->countries_id = trim($input['countries_id']);
     $user->email_new_subscriber = $input['email_new_subscriber'] ?? 'no';
     $user->save();

     \Session::flash('status', trans('auth.success_update'));

     return redirect('settings');
    }

    public function notifications()
    {
      // Notifications
      $notifications = DB::table('notifications')
         ->select(DB::raw('
        notifications.id id_noty,
        notifications.type,
        notifications.created_at,
        users.id userId,
        users.username,
        users.hide_name,
        users.name,
        users.avatar,
        updates.id,
        updates.description,
        U2.username usernameAuthor,
        messages.message
        '))
        ->leftjoin('users', 'users.id', '=', DB::raw('notifications.author'))
        ->leftjoin('updates', 'updates.id', '=', DB::raw('notifications.target'))
        ->leftjoin('messages', 'messages.id', '=', DB::raw('notifications.target'))
        ->leftjoin('users AS U2', 'U2.id', '=', DB::raw('updates.user_id'))
        ->leftjoin('comments', 'comments.updates_id', '=', DB::raw('notifications.target
        AND comments.user_id = users.id
        AND comments.updates_id = updates.id'))
        ->where('notifications.destination', '=',  auth()->user()->id)
        ->where('users.status', '=',  'active')
        ->groupBy('notifications.id')
        ->orderBy('notifications.id', 'DESC')
        ->paginate(20);

      // Mark seen Notification
      $getNotifications = Notifications::where('destination', auth()->user()->id)->where('status', '0');
      $getNotifications->count() > 0 ? $getNotifications->update(['status' => '1']) : null;

      return view('users.notifications', ['notifications' => $notifications]);
    }

    public function settingsNotifications()
    {
      $user = User::find(auth()->user()->id);
      $user->notify_new_subscriber = $this->request->notify_new_subscriber ?? 'no';
      $user->notify_liked_post = $this->request->notify_liked_post ?? 'no';
      $user->notify_commented_post = $this->request->notify_commented_post ?? 'no';
      $user->notify_new_tip = $this->request->notify_new_tip ?? 'no';
      $user->email_new_subscriber = $this->request->email_new_subscriber ?? 'no';
      $user->save();

      return response()->json([
          'success' => true,
      ]);
    }

    public function deleteNotifications()
    {
      auth()->user()->notifications()->delete();
      return back();
    }

    public function password()
    {
      return view('users.password');
    }//<--- End Method

      public function updatePassword(Request $request)
      {

  	   $input = $request->all();
  	   $id    = auth()->user()->id;
       $passwordRequired = auth()->user()->password != '' ? 'required|' : null;

  		   $validator = Validator::make($input, [
  			'old_password' => $passwordRequired.'min:6',
  	     'new_password' => 'required|min:6',
      	]);

  			if ($validator->fails()) {
           return redirect()->back()
  						 ->withErrors($validator)
  						 ->withInput();
  					 }

  	   if (auth()->user()->password != '' && !\Hash::check($input['old_password'], auth()->user()->password)) {
  		    return redirect('settings/password')->with( array( 'incorrect_pass' => trans('general.password_incorrect') ) );
  		}

  	   $user = User::find($id);
  	   $user->password  = \Hash::make($input[ "new_password"] );
  	   $user->save();

  	   \Session::flash('status',trans('auth.success_update_password'));

  	   return redirect('settings/password');

  	}//<--- End Method

    public function mySubscribers()
    {
      $subscriptions = auth()->user()->mySubscriptions()->orderBy('id','desc')->paginate(20);


      return view('users.my_subscribers')->withSubscriptions($subscriptions);
    }

    public function mySubscriptions()
    {
      $subscriptions = auth()->user()->userSubscriptions()->orderBy('id','desc')->paginate(20);
      return view('users.my_subscriptions')->withSubscriptions($subscriptions);
    }

    public function myPayments()
    {
      if (request()->is('my/payments')) {
        $transactions = auth()->user()->myPayments()->orderBy('id','desc')->paginate(20);
      } elseif (request()->is('my/payments/received')) {
        $transactions = auth()->user()->myPaymentsReceived()->orderBy('id','desc')->paginate(20);
      } else {
        abort(404);
      }

      return view('users.my_payments')->withTransactions($transactions);
    }

    public function payoutMethod()
    {
      return view('users.payout_method');
    }

    public function payoutMethodConfigure()
    {

		if( $this->request->type != 'paypal' && $this->request->type != 'bank' ) {
			return redirect('settings/payout/method');
			exit;
		}

		// Validate Email Paypal
		if( $this->request->type == 'paypal') {
			$rules = array(
	        'email_paypal' => 'required|email|confirmed',
        );

		$this->validate($this->request, $rules);

		$user                  = User::find(auth()->user()->id);
		$user->paypal_account  = $this->request->email_paypal;
		$user->payment_gateway = 'PayPal';
		$user->save();

		\Session::flash('status', trans('admin.success_update'));
		return redirect('settings/payout/method')->withInput();

		}// Validate Email Paypal

		elseif ($this->request->type == 'bank') {

			$rules = array(
	        'bank_details'  => 'required|min:20',
       		 );

		  $this->validate($this->request, $rules);

		   $user                  = User::find(auth()->user()->id);
		   $user->bank            = strip_tags($this->request->bank_details);
		   $user->payment_gateway = 'Bank';
		   $user->save();

			\Session::flash('status', trans('admin.success_update'));
			return redirect('settings/payout/method');
		}

    }//<--- End Method

    public function uploadAvatar()
		{
      $validator = Validator::make($this->request->all(), [
        'avatar' => 'required|mimes:jpg,gif,png,jpe,jpeg|dimensions:min_width=200,min_height=200|max:'.$this->settings->file_size_allowed.'',
      ]);

		   if ($validator->fails()) {
		        return response()->json([
				        'success' => false,
				        'errors' => $validator->getMessageBag()->toArray(),
				    ]);
		    }

		// PATHS
	  $path = config('path.avatar');

		 //<--- HASFILE PHOTO
	    if($this->request->hasFile('avatar'))	{

				$photo     = $this->request->file('avatar');
				$extension = $this->request->file('avatar')->getClientOriginalExtension();
				$avatar    = strtolower(auth()->user()->username.'-'.auth()->user()->id.time().str_random(10).'.'.$extension );

				set_time_limit(0);
				ini_set('memory_limit', '512M');

				$imgAvatar = Image::make($photo)->orientate()->fit(200, 200, function ($constraint) {
					$constraint->aspectRatio();
					$constraint->upsize();
				})->encode($extension);

				// Copy folder
				Storage::put($path.$avatar, $imgAvatar, 'public');

				//<<<-- Delete old image -->>>/
				if (auth()->user()->avatar != $this->settings->avatar) {
					Storage::delete(config('path.avatar').auth()->user()->avatar);
				}

				// Update Database
				auth()->user()->update(['avatar' => $avatar]);

				return response()->json([
				        'success' => true,
				        'avatar' => Helper::getFile($path.$avatar),
				    ]);
	    }//<--- HASFILE PHOTO
    }//<--- End Method Avatar

    public function settingsPage()
    {
      $genders = explode(',', $this->settings->genders);
      return view('users.edit_my_page', ['genders' => $genders]);
    }

    public function updateSettingsPage()
    {

      $input = $this->request->all();
      $id    = auth()->user()->id;
      $input['is_admin'] = $id;
      $input['is_creator'] = auth()->user()->verified_id == 'yes' ? 0 : 1;

      $messages = array (
      "letters" => trans('validation.letters'),
      "email.required_if" => trans('validation.required'),
      "birthdate.before" => trans('general.error_adult'),
      "story.required_if" => trans('validation.required'),
		);

		 Validator::extend('ascii_only', function($attribute, $value, $parameters){
    		return !preg_match('/[^x00-x7F\-]/i', $value);
		});

		// Validate if have one letter
	Validator::extend('letters', function($attribute, $value, $parameters){
    	return preg_match('/[a-zA-Z0-9]/', $value);
	});

      $validator = Validator::make($input, [
        'full_name' => 'required|string|max:100',
        'username'  => 'required|min:3|max:25|ascii_only|alpha_dash|letters|unique:pages,slug|unique:reserved,name|unique:users,username,'.$id,
        'email'  => 'required_if:is_admin,==,1|unique:users,email,'.$id,
        'website' => 'url',
        'facebook' => 'url',
        'twitter' => 'url',
        'instagram' => 'url',
        'youtube' => 'url',
        'pinterest' => 'url',
        'github' => 'url',
        'story' => 'required_if:is_creator,==,0|max:'.$this->settings->story_length.'',
        'countries_id' => 'required',
        'city' => 'max:100',
        'address' => 'max:100',
        'zip' => 'max:20',
        'birthdate' =>'required|date|before:'.Carbon::now()->subYears(18),
     ], $messages);

      if ($validator->fails()) {
           return response()->json([
               'success' => false,
               'errors' => $validator->getMessageBag()->toArray(),
           ]);
       } //<-- Validator

      $user                  = User::find($id);
      $user->name            = strip_tags($this->request->full_name);
      $user->username        = trim($this->request->username);
      $user->email           = $this->request->email ? trim($this->request->email) : auth()->user()->email;
      $user->website         = trim($this->request->website) ?? '';
      $user->categories_id   = $this->request->categories_id ?? '';
      $user->profession      = $this->request->profession;
      $user->countries_id    = $this->request->countries_id;
      $user->city            = $this->request->city;
      $user->address         = $this->request->address;
      $user->zip             = $this->request->zip;
      $user->company         = $this->request->company;
      $user->story           = trim(Helper::checkTextDb($this->request->story));
      $user->facebook         = trim($this->request->facebook) ?? '';
      $user->twitter         = trim($this->request->twitter) ?? '';
      $user->instagram         = trim($this->request->instagram) ?? '';
      $user->youtube         = trim($this->request->youtube) ?? '';
      $user->pinterest         = trim($this->request->pinterest) ?? '';
      $user->github         = trim($this->request->github) ?? '';
      $user->plan           = 'user_'.auth()->user()->id;
      $user->gender         = $this->request->gender;
      $user->birthdate      = $this->request->birthdate;
      $user->language      = $this->request->language;
      $user->hide_name     = $this->request->hide_name ?? 'no';
      $user->save();

      return response()->json([
              'success' => true,
              'url' => url(trim($this->request->username)),
              'locale' => $this->request->language != '' && config('app.locale') != $this->request->language ? true : false,
            ]);

    }//<--- End Method

    public function saveSubscription()
    {

      $input = $this->request->all();

      if (auth()->user()->verified_id == 'no' || auth()->user()->verified_id == 'reject') {
        return redirect()->back()
            ->withErrors([
     					'errors' => trans('general.error'),
     				]);
          }

      $id    = auth()->user()->id;
      $input['_verified_id'] = auth()->user()->verified_id;

      if ($this->settings->currency_position == 'right') {
				$currencyPosition =  2;
			} else {
				$currencyPosition =  null;
			}

      if ($this->request->free_subscription) {
        $priceRequired = null;
      } else {
        $priceRequired = 'required_if:_verified_id,==,yes|';
      }

      $messages = array (
			'price.min' => trans('users.price_minimum_subscription'.$currencyPosition, ['symbol' => $this->settings->currency_symbol, 'code' => $this->settings->currency_code]),
			'price.max' => trans('users.price_maximum_subscription'.$currencyPosition, ['symbol' => $this->settings->currency_symbol, 'code' => $this->settings->currency_code]),
      "price.required_if" => trans('general.subscription_price_required'),
		);

  if (auth()->user()->verified_id == 'no' || auth()->user()->verified_id == 'reject') {
    $this->settings->min_subscription_amount = 0;
  } else {
    $this->settings->min_subscription_amount = $this->settings->min_subscription_amount;
  }

      $validator = Validator::make($input, [
        'price' => $priceRequired.'numeric|min:'.$this->settings->min_subscription_amount.'|max:'.$this->settings->max_subscription_amount.'',
     ], $messages);

     if ($validator->fails()) {
        return redirect()->back()
            ->withErrors($validator)
            ->withInput();
          }

      $user                    = User::find($id);
      $user->price             = $this->request->price ?? auth()->user()->price;
      $user->free_subscription = $this->request->free_subscription ?? 'no';
      $user->plan              = 'user_'.auth()->user()->id;
      $user->save();

      // Create Plan Stripe
      if (auth()->user()->verified_id == 'yes' && ! $this->request->free_subscription) {
        $this->createPlanStripe();
      }

      // Create Plan Paystack
      if (auth()->user()->verified_id == 'yes' && ! $this->request->free_subscription) {
        $this->createPlanPaystack();
      }

      \Session::flash('status', trans('admin.success_update'));
			return redirect('settings/subscription');

    }//<--- End Method

  protected function createPlanStripe()
  {
    $payment = PaymentGateways::whereName('Stripe')->whereEnabled(1)->first();
    $plan = 'user_'.auth()->user()->id;

    if ($payment) {
      if ($this->request->price != auth()->user()->price) {
        $stripe = new \Stripe\StripeClient($payment->key_secret);

        try {
          $planCurrent = $stripe->plans->retrieve($plan, []);

          // Delete old plan
          $stripe->plans->delete($plan, []);

          // Delete Product
          $stripe->products->delete($planCurrent->product, []);
        } catch (\Exception $exception) {
          // not exists
        }

        // Create Plan
        $plan = $stripe->plans->create([
            'currency' => $this->settings->currency_code,
            'interval' => 'month',
            "product" => [
                "name" => trans('general.subscription_for').' @'.auth()->user()->username,
            ],
            'nickname' => $plan,
            'id' => $plan,
            'amount' => $this->settings->currency_code == 'JPY' ? $this->request->price : $this->request->price * 100,
        ]);
      }
    }
  }

  protected function createPlanPaystack()
  {
    $payment = PaymentGateways::whereName('Paystack')->whereEnabled(1)->first();

    if ($payment) {

      // initiate the Library's Paystack Object
      $paystack = new Paystack($payment->key_secret);

      //========== Create Plan if no exists
      if ( ! auth()->user()->paystack_plan) {

        $userPlan = $paystack->plan->create([
                'name'=> trans('general.subscription_for').' @'.auth()->user()->username,
                'amount'=> auth()->user()->price*100,
                'interval'=> 'monthly',
                'currency'=> $this->settings->currency_code
              ]);

      $planCode = $userPlan->data->plan_code;

      // Insert Plan Code to User
      User::whereId(auth()->user()->id)->update([
            'paystack_plan' => $planCode
          ]);
      } else {
        if ($this->request->price != auth()->user()->price) {

          $userPlan = $paystack->plan->update([
                  'name'=> trans('general.subscription_for').' @'.auth()->user()->username,
                  'amount'=> $this->request->price*100,
                ],['id' => auth()->user()->paystack_plan]);
        }
      }
    } // payment
  } // end method


   public function uploadCover(Request $request)
   {
     $settings  = AdminSettings::first();

     $validator = Validator::make($this->request->all(), [
       'image' => 'required|mimes:jpg,gif,png,jpe,jpeg|dimensions:min_width=800,min_height=400|max:'.$settings->file_size_allowed.'',
     ]);

      if ($validator->fails()) {
           return response()->json([
               'success' => false,
               'errors' => $validator->getMessageBag()->toArray(),
           ]);
       }

   // PATHS
   $path = config('path.cover');

    //<--- HASFILE PHOTO
     if ($this->request->hasFile('image') )	{

       $photo       = $this->request->file('image');
       $widthHeight = getimagesize($photo);
       $extension   = $photo->getClientOriginalExtension();
       $cover       = strtolower(auth()->user()->username.'-'.auth()->user()->id.time().str_random(10).'.'.$extension );

       set_time_limit(0);
       ini_set('memory_limit', '512M');

       //=============== Image Large =================//
       $width     = $widthHeight[0];
       $height    = $widthHeight[1];
       $max_width = $width < $height ? 800 : 1500;

       if ($width > $max_width) {
         $coverScale = $max_width / $width;
       } else {
         $coverScale = 1;
       }

       $scale    = $coverScale;
       $widthCover = ceil($width * $scale);

       $imgCover = Image::make($photo)->orientate()->resize($widthCover, null, function ($constraint) {
         $constraint->aspectRatio();
         $constraint->upsize();
       })->encode($extension);

       // Copy folder
       Storage::put($path.$cover, $imgCover, 'public');

       //<<<-- Delete old image -->>>/
         Storage::delete(config('path.cover').auth()->user()->cover);

       // Update Database
       auth()->user()->update(['cover' => $cover]);

       return response()->json([
               'success' => true,
               'cover' => Helper::getFile($path.$cover),
           ]);

     }//<--- HASFILE PHOTO
   }//<--- End Method Cover

    public function withdrawals()
    {
      $withdrawals = auth()->user()->withdrawals()->orderBy('id','desc')->paginate(20);

      return view('users.withdrawals')->withWithdrawals($withdrawals);
    }

    public function makeWithdrawals()
    {
      if (auth()->user()->balance >= $this->settings->amount_min_withdrawal
          && auth()->user()->payment_gateway != ''
          && Withdrawals::where('user_id', auth()->user()->id
          )
          ->where('status','pending')
          ->count() == 0) {

        if (auth()->user()->payment_gateway == 'PayPal') {
   		   	$_account = auth()->user()->paypal_account;
   		   } else {
   		   	$_account = auth()->user()->bank;
   		   }

 			$sql           = new Withdrawals;
 			$sql->user_id  = auth()->user()->id;
 			$sql->amount   = auth()->user()->balance;
 			$sql->gateway  = auth()->user()->payment_gateway;
 			$sql->account  = $_account;
 			$sql->save();

      // Remove Balance the User
      $userBalance = User::find(auth()->user()->id);
      $userBalance->balance = 0;
      $userBalance->save();

      }

      return redirect('settings/withdrawals');
    } // End Method makeWithdrawals

    public function deleteWithdrawal()
    {
  		$withdrawal = auth()->user()->withdrawals()
      ->whereId($this->request->id)
      ->whereStatus('pending')
      ->firstOrFail();

      // Add Balance the User again
      User::find(auth()->user()->id)->increment('balance', $withdrawal->amount);

			$withdrawal->delete();

			return redirect('settings/withdrawals');

    }//<--- End Method

    public function deleteImageCover()
    {
      $path  = 'public/cover/';
      $id    = auth()->user()->id;

      // Image Cover
  		$image = $path.auth()->user()->cover;

      if (\File::exists($image)) {
        \File::delete($image);
      }

      $user = User::find($id);
      $user->cover = '';
      $user->save();

      return response()->json([
              'success' => true,
          ]);
    }// End Method

    public function reportCreator(Request $request)
    {
  		$data = Reports::firstOrNew(['user_id' => auth()->user()->id, 'report_id' => $request->id]);

      $validator = Validator::make($this->request->all(), [
        'reason' => 'required|in:spoofing,copyright,privacy_issue,violent_sexual,spam,fraud,under_age',
      ]);

       if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'text' => __('general.error'),
            ]);
        }

  		if ($data->exists ) {
        return response()->json([
            'success' => false,
            'text' => __('general.already_sent_report'),
        ]);
  		} else {

  			$data->type = 'user';
        $data->reason = $request->reason;
  			$data->save();

        return response()->json([
            'success' => true,
            'text' => __('general.reported_success'),
        ]);
  		}
  	}//<--- End Method

    public function like(Request $request){

  		$like = Like::firstOrNew(['user_id' => auth()->user()->id, 'updates_id' => $request->id]);

  		$user = Updates::find($request->id);

  		if ($like->exists) {

  			   $notifications = Notifications::where('destination', $user->user_id)
  			   ->where('author', auth()->user()->id)
  			   ->where('target', $request->id)
  			   ->where('type','2')
  			   ->first();

  				// IF ACTIVE DELETE FOLLOW
  				if ($like->status == '1') {
            $like->status = '0';
  					$like->update();

            	// DELETE NOTIFICATION
  				if (isset($notifications)) {
            $notifications->status = '1';
            $notifications->update();
          }

  				// ELSE ACTIVE AGAIN
  				} else {
  					$like->status = '1';
  					$like->update();

            // ACTIVE NOTIFICATION
  					if (isset($notifications)) {
              $notifications->status = '0';
              $notifications->update();
            }
  				}

  		} else {

  			// INSERT
  			$like->save();

  			// Send Notification //destination, author, type, target
  			if ($user->user_id != auth()->user()->id && $user->user()->notify_liked_post == 'yes') {
  				Notifications::send($user->user_id, auth()->user()->id, '2', $request->id);
  			}
  		}

      $totalLike = Helper::formatNumber($user->likes()->count());

      return $totalLike;
  	}//<---- End Method

    public function ajaxNotifications()
    {
  		 if (request()->ajax()) {

         // Logout user suspended
         if (auth()->user()->status == 'suspended') {
           auth()->logout();
         }

  			// Notifications
  			$notifications_count = auth()->user()->notifications()->where('status', '0')->count();
        // Messages
  			$messages_count = auth()->user()->messagesInbox();

  			return response()->json([
          'messages' => $messages_count,
          'notifications' => $notifications_count
        ]);

  		   } else {
  				return response()->json(['error' => 1]);
  			}
     }//<---- * End Method

     public function verifyAccount()
     {
       return view('users.verify_account');
     }//<---- * End Method

     public function verifyAccountSend()
     {
       $checkRequest = VerificationRequests::whereUserId(auth()->user()->id)->whereStatus('pending')->first();

       if($checkRequest) {
         return redirect()->back()
     				->withErrors([
     					'errors' => trans('admin.pending_request_verify'),
     				]);
       } elseif (auth()->user()->verified_id == 'reject') {
         return redirect()->back()
     				->withErrors([
     					'errors' => trans('admin.rejected_request'),
     				]);
       }

       $input = $this->request->all();
       $input['isUSCitizen'] = auth()->user()->countries_id;

       $messages = [
         "form_w9.required_if" => trans('general.form_w9_required')
       ];

      $validator = Validator::make($input, [
        'address'  => 'required',
        'city' => 'required',
        'zip' => 'required',
        'image' => 'required|mimes:jpg,gif,png,jpe,jpeg,zip|max:'.$this->settings->file_size_allowed_verify_account.'',
        'form_w9'  => 'required_if:isUSCitizen,==,1|mimes:pdf|max:'.$this->settings->file_size_allowed_verify_account.'',
     ], $messages);

      if ($validator->fails()) {
          return redirect()->back()
                    ->withErrors($validator)
                    ->withInput();
      }

      // PATHS
  		$path = config('path.verification');

      if ($this->request->hasFile('image')) {

			$extension = $this->request->file('image')->getClientOriginalExtension();
			$fileImage = strtolower(auth()->user()->id.time().Str::random(40).'.'.$extension);

      $this->request->file('image')->storePubliclyAs($path, $fileImage);

	   }//<====== End HasFile

     if ($this->request->hasFile('form_w9')) {

       $extension = $this->request->file('form_w9')->getClientOriginalExtension();
       $fileFormW9 = strtolower(auth()->user()->id.time().Str::random(40).'.'.$extension);

     $this->request->file('form_w9')->storePubliclyAs($path, $fileFormW9);

    }//<====== End HasFile

      $sql          = new VerificationRequests;
 			$sql->user_id = auth()->user()->id;
 			$sql->address = $input['address'];
 			$sql->city    = $input['city'];
      $sql->zip     = $input['zip'];
      $sql->image   = $fileImage;
      $sql->form_w9 = $fileFormW9 ?? '';
 			$sql->save();

      \Session::flash('status', trans('general.send_success_verification'));

      return redirect('settings/verify/account');
     }

     public function invoice($id)
     {
       $data = Transactions::whereId($id)->where('user_id', auth()->user()->id)->whereApproved('1')->firstOrFail();

       if(auth()->user()->address == ''
         || auth()->user()->city == ''
         || auth()->user()->zip == ''
         || auth()->user()->name == ''
     ) {
       return back()->withErrorMessage('Error');
     }

   		return view('users.invoice')->withData($data);
     }

     public function formAddUpdatePaymentCard()
     {
       $payment = PaymentGateways::whereName('Stripe')->whereEnabled(1)->firstOrFail();
       \Stripe\Stripe::setApiKey($payment->key_secret);

       return view('users.add_payment_card', [
         'intent' => auth()->user()->createSetupIntent(),
         'key' => $payment->key
       ]);
     }// End Method

     public function addUpdatePaymentCard()
     {
       $payment = PaymentGateways::whereName('Stripe')->whereEnabled(1)->firstOrFail();
       \Stripe\Stripe::setApiKey($payment->key_secret);

       if (! $this->request->payment_method) {
         return response()->json([
           "success" => false
         ]);
       }

       if ( ! auth()->user()->hasPaymentMethod()) {
           auth()->user()->createOrGetStripeCustomer();
       }

       try {
         auth()->user()->deletePaymentMethods();
       } catch (\Exception $e) {
         // error
       }

       auth()->user()->updateDefaultPaymentMethod($this->request->payment_method);
       auth()->user()->save();

       return response()->json([
         "success" => true
       ]);
     }// End Method

     public function cancelSubscription($id)
     {
       $checkSubscription = auth()->user()->userSubscriptions()->whereStripeId($id)->firstOrFail();
       $creator = User::wherePlan($checkSubscription->stripe_plan)->first();
       $payment = PaymentGateways::whereName('Stripe')->whereEnabled(1)->firstOrFail();

       $stripe = new \Stripe\StripeClient($payment->key_secret);

       try {
         $response = $stripe->subscriptions->cancel($id, []);
       } catch (\Exception $e) {
         return back()->withErrorMessage($e->getMessage());
       }

       sleep(2);

       $checkSubscription->ends_at = date('Y-m-d H:i:s', $response->current_period_end);
       $checkSubscription->save();

       session()->put('subscription_cancel', trans('general.subscription_cancel'));
       return redirect($creator->username);

     }// End Method

     // Delete Account
     public function deleteAccount()
     {
       if (!\Hash::check($this->request->password, auth()->user()->password) ) {
  		    return back()->with(['incorrect_pass' => trans('general.password_incorrect')]);
  		}
       if (auth()->user()->id == 1) {
         return redirect('settings/page');
       }

       $this->deleteUser(auth()->user()->id);

       return redirect('/');
     }

     // My Bookmarks
     public function myBookmarks()
     {
       $bookmarks = auth()->user()->bookmarks()->orderBy('bookmarks.id','desc')->paginate($this->settings->number_posts_show);

       $users = $this->userExplore();

       return view('users.bookmarks', ['updates' => $bookmarks, 'users' => $users]);
     }

     // Download File
     public function downloadFile($id)
   	{
      $post = Updates::findOrFail($id);

      if ( ! auth()->user()->checkSubscription($post->user())) {
        abort(404);
      }

      $pathFile = config('path.files').$post->file;
      $headers = [
				'Content-Type:' => ' application/x-zip-compressed',
				'Cache-Control' => 'no-cache, no-store, must-revalidate',
				'Pragma' => 'no-cache',
				'Expires' => '0'
			];

      return Storage::download($pathFile, $post->file_name.' '.__('general.by').' @'.$post->user()->username.'.zip', $headers);

    }

    public function myCards()
    {
      $payment = PaymentGateways::whereName('Stripe')->whereEnabled(1)->first();
      $paystackPayment = PaymentGateways::whereName('Paystack')->whereEnabled(1)->first();

      if ( ! $payment && ! $paystackPayment) {
        abort(404);
      }

      if (auth()->user()->stripe_id != '' && auth()->user()->card_brand != '' && isset($payment->key_secret)) {
        $stripe = new \Stripe\StripeClient($payment->key_secret);

        $response = $stripe->paymentMethods->all([
          'customer' => auth()->user()->stripe_id,
          'type' => 'card',
        ]);

        $expiration = $response->data[0]->card->exp_month.'/'.$response->data[0]->card->exp_year;
      }

      $chargeAmountPaystack = ['NGN' => '50.00', 'GHS' => '0.10', 'ZAR' => '1', 'USD' => 0.20];

      if (array_key_exists($this->settings->currency_code, $chargeAmountPaystack)) {
          $chargeAmountPaystack = $chargeAmountPaystack[$this->settings->currency_code];
      } else {
          $chargeAmountPaystack = 0;
      }

      return view('users.my_cards',[
        'key_secret' => $payment->key_secret ?? null,
        'expiration' => $expiration ?? null,
        'paystackPayment' => $paystackPayment,
        'chargeAmountPaystack' => $chargeAmountPaystack
      ]);
    }

    // Privacy Security
    public function privacySecurity()
    {
      $sessions = \DB::table('sessions')
            ->where('user_id', auth()->user()->id)
            ->orderBy('id', 'DESC')
            ->first();

        return view('users.privacy_security')
                ->with('sessions', $sessions)
                ->with('current_session_id', \Session::getId());;
    }

    public function savePrivacySecurity()
    {
      $user = User::find(auth()->user()->id);
      $user->hide_profile = $this->request->hide_profile ?? 'no';
      $user->hide_last_seen = $this->request->hide_last_seen ?? 'no';
      $user->hide_count_subscribers = $this->request->hide_count_subscribers ?? 'no';
      $user->hide_my_country = $this->request->hide_my_country ?? 'no';
      $user->show_my_birthdate = $this->request->show_my_birthdate ?? 'no';
      $user->save();

			return redirect('privacy/security')->withStatus(trans('admin.success_update'));
    }

    // Logout a session based on session id.
    public function logoutSession($id)
    {

        \DB::table('sessions')
            ->where('id', $id)->delete();

        return redirect('privacy/security');
    }

    public function deletePaymentCard()
    {
      $paymentMethod = auth()->user()->defaultPaymentMethod();

      $paymentMethod->delete();

      return redirect('my/cards')->withSuccessRemoved(__('general.successfully_removed'));
    }

    public function invoiceDeposits($id)
    {
      $data = Deposits::whereId($id)->whereUserId(auth()->user()->id)->whereStatus('active')->firstOrFail();

      if (auth()->user()->address == ''
        || auth()->user()->city == ''
        || auth()->user()->zip == ''
        || auth()->user()->name == ''
    ) {
      return back()->withErrorMessage('Error');
    }

     return view('users.invoice-deposits')->withData($data);
    }

    // My Purchases
    public function myPurchases()
    {
      $purchases = auth()->user()->payPerView()->orderBy('pay_per_views.id','desc')->paginate($this->settings->number_posts_show);

      $users = $this->userExplore();

      return view('users.my-purchases', [
        'updates' => $purchases,
        'users' => $users
        ]);
    }

    // My Purchases Ajax Pagination
    public function ajaxMyPurchases()
    {
      $skip = $this->request->input('skip');
      $total = $this->request->input('total');

      $data = auth()->user()->payPerView()->orderBy('pay_per_views.id','desc')->skip($skip)->take($this->settings->number_posts_show)->get();
      $counterPosts = ($total - $this->settings->number_posts_show - $skip);

      return view('includes.updates',
          ['updates' => $data,
          'ajaxRequest' => true,
          'counterPosts' => $counterPosts,
          'total' => $total
          ])->render();

    }//<--- End Method
}
